/* Copyright (C) Martin Buchholz 2003 */

import java.io.*;
import java.util.regex.*;
import martin.Command;
import martin.CommandResults;

public class CompileErrors
{
    private static void WriteStringToFile (String filename, String contents)
	throws FileNotFoundException, IOException {
	PrintWriter pw = new PrintWriter
	    (new FileOutputStream (new File (filename)));
	pw.print (contents);
	pw.close ();
    }

    private static final String prog_    = "CompileErrors";
    private final static String prog_cc_ = prog_ + ".cc";

    private static void CheckGccErrorMessage (String fileContents,
					      String errorMsgRegex)
	throws FileNotFoundException, IOException
    {
	WriteStringToFile (prog_cc_, fileContents);
	String [] argv = new String []
	    { "g++", "-Wall",
	      "-fmessage-length=0", // Avoid unpredictable line breaks
	      "-I.",
	      prog_cc_, "Suit.cc", "-o", prog_ };

	CommandResults res = Command.run (argv);

	if (res.exception == null &&
	    res.exitVal != 0 &&
	    Pattern.compile (errorMsgRegex).matcher (res.stderr).find()) {
	    System.out.println ("PASS");
	} else {
	    System.out.print ("\nTEST FAILED:\n");
	    System.out.print ("Regex=\"" + errorMsgRegex + "\"\n");
	    System.out.print ("<program>\n" + fileContents + "</program>\n");
	    res.debugPrint ();
	}

	new File (prog_cc_).delete();
	new File (prog_).delete();
    }

    private static void StandardAppTest (String testCode, String errorMsgRegex)
	throws FileNotFoundException, IOException
    {
	CheckGccErrorMessage
	    ("#include <iostream>\n" +
	     "#include <cassert>\n"  +
	     "#include <string>\n"   +
	     "using std::cout;\n" +
	     "using std::string;\n" +

	     "#include \"Suit.hpp\"\n" +
	     "using namespace Cards;\n" +
	     "\n" +
	     testCode,
	     errorMsgRegex);
    }

    private static void MainBodyTest (String mainBody, String errorMsgRegex)
	throws FileNotFoundException, IOException
    {
	StandardAppTest ("int main ()\n{\n" + mainBody + "\n}\n",
			 errorMsgRegex);
    }

    private static void runTests ()
	throws FileNotFoundException, IOException
    {
	// SUIT_ is not (yet) private
// 	MainBodyTest ("Suit::SUIT_ s = Suit::CLUBS;",
// 		      ".*SUIT_' is private");
	MainBodyTest ("Suit s = Suit::CLUBS;",
		      "conversion .* to non-scalar type `.*Suit' requested");
	MainBodyTest ("Suit s (Suit::CLUBS);",
		      "no matching function for call");
	MainBodyTest ("Suit s = Suit::clubs; &Suit::clubs;",
		      "operator&.*is private");
	// operator new is not (yet) private
// 	MainBodyTest ("new Suit::Literal<Suit::CLUBS> x (3);",
// 		      "operator new.*is private");
	MainBodyTest ("Suit::spades + 1;",
		      "no match for `.*\\+.*' operator");
	MainBodyTest ("1 + Suit::spades;",
		      "no match for `.*\\+.*' operator");
	MainBodyTest ("static_cast<int>(Suit::spades);",
		      "invalid static_cast .* to type `int'");
	MainBodyTest ("const Suit& s = Suit::clubs;",
		      "`.*Suit' is an inaccessible base");
	MainBodyTest ("int x = Suit::clubs;",
		      "cannot convert `.*Clubs.*' to `int'");
    }

    public static void main (String []args) {
	try {
	    runTests ();
	} catch (Exception e) {
	    e.printStackTrace(); Runtime.getRuntime().exit (1);
	}
    }
}
