/* Copyright (C) Martin Buchholz 2003 */

#include <cstdlib>
#include <cassert>
#include "Bool.hpp"

// ----------------------------------------------------------------
// Test code for the Bool class.
// For tests where compile errors are expected and desired, see
// CompileErrors.java.
// ----------------------------------------------------------------
inline int foo (Bool b) { return b ? 1 : 0; }
inline int foo (bool b) { return b ? 1 : 0; }

struct S { S(Bool) {} };
inline void bar (S) {}

inline bool aborter () { abort(); return true; }
inline Bool Aborter () { abort(); return true; }

extern void g (Bool);
extern void g (int);
extern void g (bool);

void f1 () { assert (TRUE); }
void f2 ()
{
  assert (true);
  assert (bool (true));
  assert (bool (true));
  assert (! false);
  assert (! bool (false));
  assert (! bool (false));
  if (! true) abort();
  if (false)  abort();

  Bool t (TRUE);  assert (t);
  Bool f (FALSE); assert (!f);

  assert ((t ? 2 : 3) == 2);
  assert ((f ? 2 : 3) == 3);

  assert (t == t);
  assert (t == true);
  assert (t == TRUE);
  assert (t != f);
  assert (t != false);
  assert (t != FALSE);
}

// gcc optimizes this
void f3 ()
{
  Bool t (true);
  Bool f (false);

  assert (t);
  assert (! f);
}

// gcc fails to optimize this
void f4 ()
{
  Bool t (true);
  Bool f (false);

  assert (t);
  assert (! f);
  assert (t);
  assert (! f);
}

struct Convertible_voidp
{
  bool b;
  Convertible_voidp (bool x) : b (x) {}
  operator void* () const { return b ? (void *) &b : (void *) 0; }
};

struct Convertible_bool_direct
{
  bool b;
  Convertible_bool_direct (bool x) : b (x) {}
  operator bool () const { return b; }
};

struct Convertible_double
{
  bool b;
  Convertible_double (bool x) : b (x) {}
  operator double () const { return b; }
};

struct Convertible_int
{
  bool b;
  Convertible_int (bool x) : b (x) {}
  operator int () const { return b; }
};

struct Convertible_PM
{
  bool b;
  Convertible_PM (bool x) : b (x) {}
  struct S_ { int M_; };
  operator int S_::* () const { return b ? & S_::M_ : 0; }
};

void f_bool (bool) {}

void f_Bool (Bool) {}

void
test1 ()
{
  // Invalid explicit conversions...
  //{ Bool b (Convertible_voidp       (true)); }
  //{ Bool b (Convertible_bool_direct (true)); }
  //{ Bool b (Convertible_double      (true)); }
  //{ Bool b (Convertible_int         (true)); }
  //{ Bool b (42); }
  //{ Bool b (0); }
  //{ Bool b (42.3); }

  // Valid explicit conversions...
  { Bool b (Convertible_PM          (true)); assert (b); }

  // Invalid implicit conversions...
  //f_Bool (Convertible_voidp       (true)); // No suitable conversion
  //f_Bool (Convertible_int         (true)); // No suitable conversion
  //f_Bool (Convertible_double      (true)); // No suitable conversion
  //f_Bool (Convertible_bool_direct (true)); // No suitable conversion
  //f_Bool (Convertible_PM          (true)); // No suitable conversion
  //f_Bool (42);

  f_Bool (true);
  f_Bool (false);

  f_Bool (Bool (true));
  f_Bool (Bool (false));

  f_Bool (TRUE);
  f_Bool (FALSE);

  f_bool (Bool (true));
  f_bool (Bool (false));

  f_bool (TRUE);
  f_bool (FALSE);

  assert (foo (true));
  assert (foo (TRUE));
  assert (foo (Convertible_int (true)));
  assert (foo (Convertible_double (true)));
  assert (foo (Convertible_bool_direct (true)));
  assert (foo (Convertible_voidp (true)));

  assert (TRUE);
  assert (Bool (TRUE));
  assert (Bool (true));
  assert (! FALSE);
  assert (! Bool (FALSE));
  assert (! Bool (false));

  if (! TRUE) abort();
  if (FALSE)  abort();

  if (TRUE)    ; else abort();
  if (! FALSE) ; else abort();

  bar (Bool (TRUE));
  bar (TRUE);

  Bool t (TRUE);  assert (t);
  Bool f (FALSE); assert (!f);

  assert ((t ? 2 : 3) == 2);
  assert ((f ? 2 : 3) == 3);

  assert ( t || aborter());
  assert (!f || aborter());
  assert (! ( f && aborter()));
  assert (! (!t && aborter()));

  assert ( t || Aborter());
  assert (!f || Aborter());
  assert (! ( f && Aborter()));
  assert (! (!t && Aborter()));

  bool bt = t; assert (bt);
  bool bf = f; assert (! bf);

  assert (t == t);
  assert (t == true);
  assert (t == TRUE);
  assert (t != f);
  assert (t != false);
  assert (t != FALSE);

  assert (TRUE == t);
  assert (TRUE == true);
  assert (TRUE == TRUE);
  assert (TRUE != f);
  assert (TRUE != false);
  assert (TRUE != FALSE);

  assert (!TRUE == !t);
  assert (!TRUE == !true);
  assert (!TRUE == !TRUE);
  assert (!TRUE != !f);
  assert (!TRUE != !false);
  assert (!TRUE != !FALSE);

}

void
test2 ()
{
  assert (true);
  assert (bool (true));
  assert (bool (true));
  assert (! false);
  assert (! bool (false));
  assert (! bool (false));
  if (! true) abort();
  if (false)  abort();

  bool t (true);  assert (t);
  bool f (false); assert (!f);

  assert ((t ? 2 : 3) == 2);
  assert ((f ? 2 : 3) == 3);

  assert ((TRUE  ? 2 : 3) == 2);
  assert ((FALSE ? 2 : 3) == 3);

  assert ( t || aborter());
  assert (!f || aborter());
  assert (! ( f && aborter()));
  assert (! (!t && aborter()));

  assert ( t || Aborter());
  assert (!f || Aborter());
  assert (! ( f && Aborter()));
  assert (! (!t && Aborter()));

  bool bt = t; assert (bt);
  bool bf = f; assert (! bf);
}

enum Type { Type_int, Type_bool, Type_Bool };

Type int_Bool (int)  { return Type_int; }
Type int_Bool (Bool) { return Type_Bool; }

void test3 ()
{
  assert (int_Bool (5) == Type_int);
  assert (int_Bool (TRUE) == Type_Bool);
  assert (int_Bool (Bool (true)) == Type_Bool);
  assert (int_Bool (true) == Type_int); // BAD!!!
}

namespace detail
{
  // ----------------------------------------------------------------
  // A general-purpose template parameter constraint class
  template <class T, bool condition> struct Constraint;
  template <class T> struct Constraint<T,true> { typedef int Int; };

  template <typename T>
  struct Convertible_to_Bool
  {
    enum { value = (boost::is_same<T,bool>::value ||
		    (boost::is_convertible<T,bool>::value &&
		     ! boost::is_convertible<T,int>::value &&
		     ! boost::is_convertible<T,void*>::value)) };
  };

  template <typename T>
  struct Convertible_to_int
  {
    enum { value = (! Convertible_to_Bool<T>::value &&
		    boost::is_convertible<T,int>::value) };
  };
}

// Template_int_Bool (int)
template <typename T>
int Template_int_Bool (T x, typename detail::Constraint<T,
		       detail::Convertible_to_int<T>::value>::Int = 0)
{
  return int (x);
}

// Template_int_Bool (Bool)
template <typename T>
Bool Template_int_Bool (T x, typename detail::Constraint<T,
			detail::Convertible_to_Bool<T>::value>::Int = 0)
{
  return Bool (x);
}

void test4 ()
{
  assert (Template_int_Bool (5) == 5);
  assert (Template_int_Bool ('5') == '5');
  // assert (Template_int_Bool (5.3) == 5); // Legal, but generates warning
  assert (Template_int_Bool (TRUE) == TRUE);
  assert (Template_int_Bool (FALSE) == false);
  assert (Template_int_Bool (Bool (true)) == true);
  assert (Template_int_Bool (Bool (false)) == FALSE);
  assert (Template_int_Bool (true) == true);
  assert (Template_int_Bool (false) == FALSE);
  assert (Template_int_Bool (Convertible_PM (true)) == TRUE);
}

struct Convertible_From_Bool { Convertible_From_Bool (Bool) {} };
void f_Convertible_From_Bool (Convertible_From_Bool) {}

void test5 ()
{
  f_Convertible_From_Bool (Bool (true));
  f_Convertible_From_Bool (Bool (false));
  f_Convertible_From_Bool (TRUE);
  f_Convertible_From_Bool (FALSE);
}

int main (int argc, char *argv[])
{
  test1 ();
  test2 ();
  test3 ();
  test4 ();
  test5 ();
  f2 ();
  f3 ();
  f4 ();
  return 0;
}
