/* Copyright (C) Martin Buchholz 2003 */

import java.io.*;
import java.util.regex.*;
import martin.Command;
import martin.CommandResults;

public class CompileErrors
{
    private static final String standard_prelude_ =
	"#include <iostream>\n" +
	"#include <cassert>\n"  +
	"#include <string>\n"   +
	"#include \"Bool.hpp\"\n" +
	"using std::cout;\n" +
	"using std::string;\n" +
	"void f_Bool (Bool) {}\n" +
	"struct Operator_voidp  { operator void*  () const { return 0;     } };\n" +
	"struct Operator_bool   { operator bool   () const { return false; } };\n" +
	"struct Operator_int    { operator int    () const { return 7;     } };\n" +
	"struct Operator_double { operator double () const { return 0.1;   } };\n" +
	"\n";

    private static final String prog_    = "CompileErrors";
    private static final String prog_cc_ = prog_ + ".cc";

    private static void WriteStringToFile (String filename, String contents)
	throws FileNotFoundException, IOException {
	PrintWriter pw = new PrintWriter
	    (new FileOutputStream (new File (filename)));
	pw.print (contents);
	pw.close ();
    }

    private static void CheckGccErrorMessage (String fileContents,
					      String errorMsgRegex)
	throws FileNotFoundException, IOException
    {
	WriteStringToFile (prog_cc_, fileContents);
	String [] argv = new String []
	    { "g++", "-Wall",
	      "-fmessage-length=0", // Avoid unpredictable line breaks
	      "-I.", "-I../../boost_1_29_0",
	      prog_cc_,
	      "-o", prog_ };

	CommandResults res = Command.run (argv);

	if (res.exception == null &&
	    res.exitVal != 0 &&
	    Pattern.compile (errorMsgRegex).matcher (res.stderr).find()) {
	    System.out.println ("PASS");
	} else {
	    System.out.println ("\nTEST FAILED:");
	    System.out.println ("Regex=\"" + errorMsgRegex + "\"");
	    res.debugPrint ();
	}

	new File (prog_cc_).delete();
	new File (prog_).delete();
    }

    private static void StandardAppTest (String testCode, String errorMsgRegex)
	throws FileNotFoundException, IOException
    {
	StringBuffer s = new StringBuffer (standard_prelude_);
	s.append (testCode);
	CheckGccErrorMessage (s.toString(), errorMsgRegex);
    }

    private static void MainBodyTest (String mainBody, String errorMsgRegex)
	throws FileNotFoundException, IOException
    {
	StandardAppTest ("int main (int argc, char *argv[])\n{\n" +
			 mainBody +
			 "\nreturn 0;\n}\n",
			 errorMsgRegex);
    }

    private static void ImplicitConversionTest (String arg, String type)
	throws FileNotFoundException, IOException
    {
	MainBodyTest ("f_Bool (" + arg + ");",
		      "conversion from `" +
		      type +
		      "' to non-scalar type `.*Bool' requested");
    }

    private static void runTests ()
	throws FileNotFoundException, IOException
    {
	MainBodyTest ("Bool x (3);",   "no matching function for call");
	MainBodyTest ("Bool x (0);",   "no matching function for call");
	MainBodyTest ("Bool x (3.9);", "no matching function for call");
	MainBodyTest ("Bool x ('x');", "no matching function for call");

	MainBodyTest ("Bool x = 3;",   "conversion from.*to non-scalar type `.*Bool'");
	MainBodyTest ("Bool x = 0;",   "conversion from.*to non-scalar type `.*Bool'");
	MainBodyTest ("Bool x = 3.9;", "conversion from.*to non-scalar type `.*Bool'");
	MainBodyTest ("Bool x = 'x';", "conversion from.*to non-scalar type `.*Bool'");

	ImplicitConversionTest ("1",   "int");
	ImplicitConversionTest ("0",   "int");
	ImplicitConversionTest ("1.1", "double");
	ImplicitConversionTest ("'z'", "char");
	ImplicitConversionTest ("Operator_voidp  ()", "Operator_voidp");
	ImplicitConversionTest ("Operator_int    ()", "Operator_int");
	ImplicitConversionTest ("Operator_bool   ()", "Operator_bool");
	ImplicitConversionTest ("Operator_double ()", "Operator_double");

	MainBodyTest ("1 == Bool (true);",  "no match for `.*' operator");
	MainBodyTest ("Bool (false) == 1;", "no match for `.*' operator");
	MainBodyTest ("TRUE == 1;",         "no match for `.*' operator");
	MainBodyTest ("1 == FALSE;",        "no match for `.*' operator");

	MainBodyTest ("1 != Bool (true);",  "no match for `.*' operator");
	MainBodyTest ("Bool (false) != 1;", "no match for `.*' operator");
	MainBodyTest ("TRUE != 1;",         "no match for `.*' operator");
	MainBodyTest ("1 != FALSE;",        "no match for `.*' operator");

	MainBodyTest ("int x = Bool (true);",    "invalid conversion from ");
	MainBodyTest ("int x = TRUE;",           "invalid conversion from ");
	MainBodyTest ("void* p = Bool (false);", "invalid conversion from ");
	MainBodyTest ("void* p = FALSE;",        "invalid conversion from ");

	MainBodyTest ("int x (Bool (true));",    "invalid conversion from ");
	MainBodyTest ("int x (TRUE);",           "invalid conversion from ");
	MainBodyTest ("void* p (Bool (false));", "invalid conversion from ");
	MainBodyTest ("void* p (FALSE);",        "invalid conversion from ");

	MainBodyTest ("& TRUE;",  "operator&.* is private");
	MainBodyTest ("& FALSE;", "operator&.* is private");

	MainBodyTest ("TRUE  = TRUE;",  "operator=.* is private");
	MainBodyTest ("FALSE = FALSE;", "operator=.* is private");

	MainBodyTest ("Bool::True& p  = new Bool::True;",  "operator new.* is private");
	MainBodyTest ("Bool::False& p = new Bool::False;", "operator new.* is private");

    }

    public static void main (String []args) {
	try {
	    runTests ();
	} catch (Exception e) {
	    e.printStackTrace(); Runtime.getRuntime().exit (1);
	}
    }
}
