/* Copyright (C) Martin Buchholz 2003 */

#ifndef Bool_hpp_INCLUDED_
#define Bool_hpp_INCLUDED_

#include <boost/type_traits/conversion_traits.hpp>
#include <boost/type_traits/same_traits.hpp>

class Bool
{
private:
  const bool val_;

  // ----------------------------------------------------------------
  // An artificial class to implement template parameter constraints
  template <bool condition, typename type = int> struct Constraint;
  template <typename type> struct Constraint<true, type>
  {
    typedef type Type;
  };

  // ----------------------------------------------------------------
  // Allow implicit conversions from bool, and _only_ from bool.
  // Template constructors are not chained with standard conversions,
  // so no implicit conversions from int, for example, are allowed.
  template <typename T>
  struct ImplicitlyConvertible
  {
    enum { value = boost::is_same<T,bool>::value };
  };

  // ----------------------------------------------------------------
  // Allow explicit conversions from any type that, like Bool itself,
  // is convertible to bool, but not to int or void*.
  template <typename T>
  struct ExplicitlyConvertible
  {
    enum { value = (! ImplicitlyConvertible<T>::value &&
		    (boost::is_convertible<T,bool>::value &&
		     ! boost::is_convertible<T,int>::value &&
		     ! boost::is_convertible<T,void*>::value)) };
  };

public:
  template <typename T>
  Bool (T x, typename Constraint<ImplicitlyConvertible<T>::value>::Type = 0)
    : val_ (x)
  {}

  template <typename T>
  explicit
  Bool (T x, typename Constraint<ExplicitlyConvertible<T>::value>::Type = 0)
    : val_ (x)
  {}


  // ----------------------------------------------------------------
  // Define True and False classes, for use with Class Literal Idiom
private:
  // Dummy constructor used by True and False
  Bool () : val_ (/* value unused */ false) {}
  class Literal;

public:
  class True;  Bool (const True &) : val_ (true ) {}
  class False; Bool (const False&) : val_ (false) {}


  // ----------------------------------------------------------------
  // Implement typesafe conversion Bool ==> bool
  // See "C++ Templates" by Josuttis and Vandevoorde
private:
  struct S_ { int M_; };
protected:
  typedef int S_::* bool_;
  inline bool_ true_  () const { return & S_::M_; }
  inline bool_ false_ () const { return 0; }
public:
  operator bool_  () const { return val_ ? true_ () : false_(); }


  // ----------------------------------------------------------------
  // Reimplement all operators defined for pointers to members
public:
  Bool operator! () const { return Bool (!val_); }

  template <typename T1, typename T2>
  struct Comparable
  {
    enum { value = (boost::is_convertible<T1,Bool>::value &&
		    boost::is_convertible<T2,Bool>::value) };
  };
  
  template <typename T1, typename T2>
  friend inline typename Constraint<Comparable<T1,T2>::value,Bool>::Type
  operator== (T1 x, T2 y) { return bool (x) == bool (y); }

  template <typename T1, typename T2>
  friend inline typename Constraint<Comparable<T1,T2>::value,Bool>::Type
  operator!= (T1 x, T2 y) { return bool (x) != bool (y); }
};

class Bool::Literal : protected Bool
{
protected:
  Literal () {}
private:
  // Prohibit nonsensical operations
  void* operator new (size_t);
  void  operator delete (void*);
  void* operator& ();
  Literal& operator= (const Literal&);
};

class Bool::True : private Literal
{
public:
  operator bool_ () const { return true_(); }
  inline False operator! () const;
};

class Bool::False : private Literal
{
public:
  operator bool_ () const { return false_(); }
  inline True operator! () const;
};

// Bool literals
// The objects themselves are unused, and can be deleted by a good linker.
extern Bool::True  TRUE;
extern Bool::False FALSE;

inline Bool::False Bool::True ::operator! () const { return FALSE; }
inline Bool::True  Bool::False::operator! () const { return TRUE;  }

#endif // Recursive Inclusion Guard
