// alignof library test suite

// Copyright (C) 2003 Martin Buchholz
//
// Permission to copy, use, sell and distribute this software is granted
// provided this copyright notice appears in all copies.
// Permission to modify the code and to distribute modified code is granted
// provided this copyright notice appears in all copies, and a notice
// that the code was modified is included with the copyright notice.
//
// This software is provided "as is" without express or implied warranty,
// and with no claim as to its suitability for any purpose.
//

// ----------------------------------------------------------------
// This code is known to work with the following compilers:
// Linux x86 g++ 2.95
// Linux x86 g++ 3.2.1
// Linux x86 g++ 3.3 20030122 pre-release
// Linux x86 g++ 3.4 20030122 pre-release
// Linux x86 Intel compiler Version 7.0
// Windows x86 MS Visual C++ .NET Version 13.00

// To work with MS Visual C++, we avoid partial template specialization.
// ----------------------------------------------------------------

#include <typeinfo>
#include <cassert>
#include <iostream>
#include <string>

#ifdef __GNUC__
#if __GNUC__ >= 3
#define HAVE_CXA_DEMANGLE
#include <cxxabi.h>
#endif
#endif

#include "alignof.h"

template <typename T>
struct TypeInfo
{
  static std::string name ()
  {
#ifdef HAVE_CXA_DEMANGLE
    char buf[256];
    size_t size = sizeof (buf);
    int status;
    abi::__cxa_demangle (typeid (T).name(), buf, &size, &status);
    return std::string (buf);
#else
    return std::string (typeid (T).name());
#endif
  }
};

template <int index>
void
check_types (char *bare_type_name)
{
#if defined (HAVE_CXA_DEMANGLE) || defined (_MSC_VER)
  using std::cout;
  using std::endl;
  using std::string;
  using Alignment::POD::types;
  using Alignment::POD::TYPES_COUNT;
  string bare = TypeInfo<typename types<index>::Type>::name();
  string ptrd = TypeInfo<typename types<index+TYPES_COUNT>::Type>::name();
  //cout << index << ' ' << bare << ' ' << ptrd << endl;
  assert (bare == bare_type_name);
#if defined __GNUC__
  assert (ptrd == string (bare_type_name) + "*");
#elif defined (_MSC_VER)
  assert (ptrd == string (bare_type_name) + " *");
#endif
#endif
}

class DI { DI () {} double d; int i; };
class DS { DS () {} double d; };
class X { X () {} int i; char c; };

int
main (int argc, char *argv[])
{
  using std::cout;
  using std::endl;
  using std::string;
  using Alignment::POD::types;
  using Alignment::POD::TYPES_COUNT;

  cout << "-----------------------------------------------------------\n";
#define CHECK_ALIGNOF_LE_SIZEOF(type) \
assert (ALIGNOF (type) <= sizeof (type))
  CHECK_ALIGNOF_LE_SIZEOF (char);
  CHECK_ALIGNOF_LE_SIZEOF (short);
  CHECK_ALIGNOF_LE_SIZEOF (int);
  CHECK_ALIGNOF_LE_SIZEOF (long);
  CHECK_ALIGNOF_LE_SIZEOF (float);
  CHECK_ALIGNOF_LE_SIZEOF (double);
  CHECK_ALIGNOF_LE_SIZEOF (long double);
  CHECK_ALIGNOF_LE_SIZEOF (void *);
  CHECK_ALIGNOF_LE_SIZEOF (void (*)());
  CHECK_ALIGNOF_LE_SIZEOF (X (X::*)());
  CHECK_ALIGNOF_LE_SIZEOF (DI);
  CHECK_ALIGNOF_LE_SIZEOF (DS);
  CHECK_ALIGNOF_LE_SIZEOF (X);

#define CHECK_ALIGNOF_DIVIDES_SIZEOF(type) \
assert ((sizeof (type) % ALIGNOF (type)) == 0)
  CHECK_ALIGNOF_DIVIDES_SIZEOF (char);
  CHECK_ALIGNOF_DIVIDES_SIZEOF (short);
  CHECK_ALIGNOF_DIVIDES_SIZEOF (int);
  CHECK_ALIGNOF_DIVIDES_SIZEOF (long);
  CHECK_ALIGNOF_DIVIDES_SIZEOF (float);
  CHECK_ALIGNOF_DIVIDES_SIZEOF (double);
  CHECK_ALIGNOF_DIVIDES_SIZEOF (long double);
  CHECK_ALIGNOF_DIVIDES_SIZEOF (void *);
  CHECK_ALIGNOF_DIVIDES_SIZEOF (void (*)());
  CHECK_ALIGNOF_DIVIDES_SIZEOF (X (X::*)());
  CHECK_ALIGNOF_DIVIDES_SIZEOF (DI);
  CHECK_ALIGNOF_DIVIDES_SIZEOF (DS);
  CHECK_ALIGNOF_DIVIDES_SIZEOF (X);

#define PRINT_ALIGNOF(type) \
cout << "ALIGNOF (" #type ") == " << ALIGNOF(type)  << endl
  PRINT_ALIGNOF (char);
  PRINT_ALIGNOF (short);
  PRINT_ALIGNOF (int);
  PRINT_ALIGNOF (long);
  PRINT_ALIGNOF (float);
  PRINT_ALIGNOF (double);
  PRINT_ALIGNOF (long double);
  PRINT_ALIGNOF (void *);
  PRINT_ALIGNOF (void (*)());
  PRINT_ALIGNOF (DI);
  PRINT_ALIGNOF (DS);
  PRINT_ALIGNOF (X);

#define PRINT_ALIGN_POD(type) \
cout << "ALIGN_POD_TYPE (" #type ") ==> " \
<< TypeInfo<ALIGN_POD_TYPE(type)>::name() << endl
  PRINT_ALIGN_POD (char);
  PRINT_ALIGN_POD (short);
  PRINT_ALIGN_POD (int);
  PRINT_ALIGN_POD (long);
  PRINT_ALIGN_POD (float);
  PRINT_ALIGN_POD (double);
  PRINT_ALIGN_POD (long double);
  PRINT_ALIGN_POD (char *);
  PRINT_ALIGN_POD (long *);
  PRINT_ALIGN_POD (void *);
  PRINT_ALIGN_POD (void (*)());
  PRINT_ALIGN_POD (DI);
  PRINT_ALIGN_POD (DS);
  PRINT_ALIGN_POD (X);

  //   cout << TypeInfo<types< 0>::Type>::name() << endl;
  //   cout << TypeInfo<types<10>::Type>::name() << endl;
  //   cout << TypeInfo<types<20>::Type>::name() << endl;
  //   cout << TypeInfo<types< 1>::Type>::name() << endl;
  //   cout << TypeInfo<types<11>::Type>::name() << endl;
  //   cout << TypeInfo<types<21>::Type>::name() << endl;
  //   cout << TypeInfo<types< 9>::Type>::name() << endl;
  //   cout << TypeInfo<types<19>::Type>::name() << endl;
  //   cout << TypeInfo<types<29>::Type>::name() << endl;

  //   using Alignment::POD::equally_aligned;
  //   cout << equally_aligned<char,int>::value << endl;
  //   cout << equally_aligned<void*,int>::value << endl;

  check_types<0>("char");
  check_types<1>("short");
  check_types<2>("int");
  check_types<3>("long");
  check_types<4>("float");
  check_types<5>("double");
  check_types<6>("long double");

#ifdef __INTEL_COMPILER
  assert (typeid(types<0>            ::Type).name() == string ("c"));
  assert (typeid(types<0+TYPES_COUNT>::Type).name() == string ("Pc"));
  assert (typeid(types<1>            ::Type).name() == string ("s"));
  assert (typeid(types<1+TYPES_COUNT>::Type).name() == string ("Ps"));
#endif
}
