/* Copyright (C) Martin Buchholz 2003 */

/* Demo/Test suite for MObS */

#include <cstdio>
#include <cassert>
#include <iostream>
#include <string>
#include <boost/type_traits/conversion_traits.hpp>
#include <boost/type_traits/same_traits.hpp>
#include "Object.hpp"
#include "NullType.hpp"
#include "Cons.hpp"
#include "lispy.hpp"
#include "Number.hpp"
#include "Int.hpp"
#include "Double.hpp"
#include "IO.hpp"
#include "toString.hpp"
#include "Number-arithmetic.hpp"
#include "IArray.hpp"
#include "Array.hpp"
#include "String.hpp"
#include "Boxed.hpp"
#include "BoxedBool.hpp"

using std::cout;
using std::string;
using boost::is_same;
using namespace MObS;

// Explicit template instantiations
// Use gcc extension to avoid instantiating unneeded
// (or uncompilable!) methods.
#ifdef __GNUC__
#define INLINE_TEMPLATE inline template
#else
#define INLINE_TEMPLATE template
#endif
// INLINE_TEMPLATE class Array<int>::Implementation;
INLINE_TEMPLATE class ObI::Impl<Array<int> >;
//INLINE_TEMPLATE class Array<int>;
// INLINE_TEMPLATE class Array<Object>::Implementation;
//INLINE_TEMPLATE class Array<Object>;
INLINE_TEMPLATE class ObI::Impl<Array<Object> >;

// ----------------------------------------------------------------

#define PRINTLN(expr) do { cout << #expr << " ==> "; println (expr); } while (0)

// template <typename T, typename U> struct ret;
// template <> struct ret<int,Double> { typedef Double type; };
// template <> struct ret<double,Number> { typedef Number type; };

// template <typename T, typename U>
// inline typename ret<T,U>::type
// foo (T t, U u) { return t + u; }

static void BinaryOps_tests ()
{
  {
    using namespace ObI::NumberOps;
    assert ((is_same<ReturnType<Int,Int,Adder>::type,Int>::value));
  }
  assert (Int (3) + Int (4) == Int (7));
  assert (3 + Int (4) == 7);
  assert (3 + Number (4) == 7);
  assert (Int (3) + 4 == 7);
  assert (Number (3) + 4 == 7);
  assert (Int (3) + 4.4 == 7.4);

  assert (Object (2.1) * Int (3) == Double (2.1 * 3));
  assert (Object (2.5) * Int (3) == 7.5);
  assert (Object (2.5) * Double (.5) == 1.25);
  assert (Object (2.5) * Number (3) == 7.5);
  assert (Object (2.5) * Number (.5) == 1.25);
  assert (Object (2.5) * Object (3) == 7.5);
  assert (Object (2.5) * Object (.5) == 1.25);

  assert (Object (2) * Int (3) == 6);
  assert (Object (2) * Double (.25) == .5);
  assert (Object (2) * Number (3) == 6);
  assert (Object (2) * Number (.25) == .5);
  assert (Object (2) * Object (3) == 6);
  assert (Object (2) * Object (.25) == .5);

  assert (Number (2.1) * Int (3) == Double (2.1 * 3));
  assert (Number (2.5) * Int (3) == 7.5);
  assert (Number (2.5) * Double (.5) == 1.25);
  assert (Number (2.5) * Number (3) == 7.5);
  assert (Number (2.5) * Number (.5) == 1.25);
  assert (Number (2.5) * Object (3) == 7.5);
  assert (Number (2.5) * Object (.5) == 1.25);

  assert (Number (2) * Int (3) == 6);
  assert (Number (2) * Double (.25) == .5);
  assert (Number (2) * Number (3) == 6);
  assert (Number (2) * Number (.25) == .5);
  assert (Number (2) * Object (3) == 6);
  assert (Number (2) * Object (.25) == .5);

  assert (Number (7) - Int (3) == 4);
  assert (8 - Int (2) == 6);
  assert (Object (9) - 5 == 4);

#if INT_DIVISION_PROMOTES
//   PredefinedOpReturnType<Int,Int,ObI::Divider>::type xy (3.3);
//   ObI::Divider::ReturnType<Int,Int>::type xyz (4.2);
//   assert ((is_same<PredefinedOpReturnType<Int,Int,ObI::Divider>::type,Double>::value));
  // Failed experiment?
  assert (Int (1) / Int (2) == 0.5);
  assert (Number (1) / Int (2) == 0.5);
  assert (1 / Int (2) == 0.5);
  assert (Int (1) / 2 == 0.5);
  assert (Int (1) / Object (2) == 0.5);
#else
  assert (Int (1) / Int (2) == 0);
  assert (Number (1) / Int (2) == 0);
  assert (1 / Int (2) == 0);
  assert (Int (1) / 2 == 0);
  assert (Int (1) / Object (2) == 0);
#endif /* INT_DIVISION_PROMOTES */
}

static void Array_tests ()
{
  Array<int> aa (3);
  assert (arrayp (aa));
  assert (sequencep (aa));
  assert (aa.length() == 3);
  assert (length (aa) == 3);
  assert (arrayp (Object (aa)));
  assert (length (Object (aa)) == 3);

  try {
    assert (length (Object (3)) == 3);
    println ("Expected exception didn't happen\n"); abort();
  } catch (Bad_Dynamic_Cast exc) {
    assert (std::string (exc.what())
	    == "Bad Dynamic Cast from Int to ISequence");
  } catch (...) {
    println ("Unexpected exception\n"); abort();
  }

  assert (sequencep (Object (aa)));
  assert (sequencep (ISequence (aa)));
  assert (typep<Array<int> > (aa));
  aa.aset (1, 9);
  assert ((aa[0] = 11) == 11);
  assert (aa.aref (0) == aa[0]);
  assert (aa.aref (0) == 11);
  assert (toString (aa) == "[11, 9, 0]");
  assert (toString (Object (aa)) == toString (aa));

  Array<Object> a2 (7, null);
  assert (arrayp (a2));
  assert (typep<Array<Object> > (a2));
  a2[0] = String ("foo");
  a2[1] = true;  assert (  Unbox<bool> (a2[1]));
  a2[2] = false; assert (! Unbox<bool> (a2[2]));
  a2[3] = aa;
  a2[4] = Double (3.5);
  a2[5] = 'x';   assert (Unbox<char>(a2[5]) == 'x');
  println (a2);
  assert (length (a2[0]) == 3);

  Array<ISequence> seqs (4, String ("?"));
  seqs [0] = String ("foo"); // OK - a String is a Sequence
  seqs [1] = Array<int> (5); // OK - an Array is a Sequence
  // seqs [2] = Int (3);     // COMPILE ERROR - an Int is not a Sequence
  // seqs [2] = Object (3);  // COMPILE ERROR - upcasts must be explicit
  //seqs [2] = ISequence (Object (3)); // RUNTIME EXCEPTION - cast failed

  // Runtime polymorphism
  assert (seqs.length () == 4);
  assert (seqs[0].length () == 3);
  assert (seqs[1].length () == 5);
}

static void is_ObType_tests ()
{
  assert (! is_ObType<int>::value);
  assert (! is_ObType<double>::value);
  assert (is_ObType<Object>::value);
  assert (is_ObType<Number>::value);
  assert (is_ObType<Double>::value);
  assert (is_ObType<Int>::value);
  assert (is_ObType<Cons>::value);
  assert (is_ObType<String>::value);
  assert (is_ObType<Array<int> >::value);
}

static void is_className_tests ()
{
#define CheckDynamicClassName(ob, s) \
assert (Object (ob).getClass().getName() == s)

  CheckDynamicClassName (Number (1), "Int");
  CheckDynamicClassName (Number (5.5), "Double");
  CheckDynamicClassName (Object (false), "Boxed<bool>");
  CheckDynamicClassName (Cons (nil, nil), "Cons");
  CheckDynamicClassName (null, "NullType");
  CheckDynamicClassName (Array<int>(3), "Array<i>"); // Compiler-dependent
  CheckDynamicClassName (String ("foo"), "String");

#define CheckStaticClassName(type, s) \
assert (StaticClass<type >::getName() == s)

  CheckStaticClassName (Object, "Object");
  CheckStaticClassName (NullType, "NullType");
  CheckStaticClassName (Boxed<bool>, "Boxed<bool>");
  CheckStaticClassName (Cons, "Cons");
  CheckStaticClassName (IArray, "IArray");
  CheckStaticClassName (String, "String");
  CheckStaticClassName (ISequence, "ISequence");
  CheckStaticClassName (Int, "Int");
  CheckStaticClassName (Double, "Double");
  CheckStaticClassName (Number, "Number");
  CheckStaticClassName (Array<int>, "Array<i>"); // Compiler-dependent
}

static void static_typep_tests ()
{
  Int i (1);
  Int i2 (2);
  Double d (1.2);
  Double d2 (3.4);
  Number ni (i);
  Object oi (i);
  Number nd (d);
  Object od (d);

  assert (static_typep<Int> (i + i2));
  assert (static_typep<Double> (i + d));
  assert (static_typep<Double> (d2 + d));
  assert (static_typep<Number> (ni + d));
  assert (static_typep<Number> (ni + i));
  assert (static_typep<Number> (ni + nd));
  assert (static_typep<Number> (oi + od));
  assert (static_typep<Number> (oi + nd));

  assert (static_typep<Int> (i + 3));
  assert (static_typep<Double> (i + 3.3));
  assert (static_typep<Double> (d2 + 3.3));
  assert (static_typep<Number> (ni + 3.4));
  assert (static_typep<Number> (ni + 7));
  assert (static_typep<Number> (ni + 7.8));
  assert (static_typep<Number> (oi + 3.32));
  assert (static_typep<Number> (oi + .3));

  assert (static_typep<Int> (3 + i2));
  assert (static_typep<Double> (3 + d));
  assert (static_typep<Double> (.5 + d));
  assert (static_typep<Number> (4 + d));
  assert (static_typep<Number> (2 + i));
  assert (static_typep<Number> (1 + nd));
  assert (static_typep<Number> (4 + od));
  assert (static_typep<Number> (9 + nd));

  assert (static_typep<Number> (Number (3)));
  assert (static_typep<Number> (Int (3)));
  assert (not static_typep<Int> (Number (3)));
  assert (not static_typep<Int> (Number (3.3)));
}

static void Cons_tests ()
{
  Number ni (3);
  Number nd (3.34);
  Cons z (ni, nd);
  Object oz (z);
  assert (consp (z));
  assert (Consp (z));
  // consp (ni); // COMPILE ERROR
  // cdr (ni);   // COMPILE ERROR

  assert (consp (oz) && Consp (oz) && !numberp (oz));
  assert (! consp (Object (ni)) && ! Consp (Object (nd)));

  assert (Int (car (z)).value() == Int (ni).value());
  assert (car (z) == ni);
  assert (Car (z) == ni);
  assert (not (car (z) == nd));
  assert (Double (cdr (z)).value() == Double (nd).value());
  assert (cdr (z) == nd);
  assert (Cdr (z) == nd);
  assert (Int (car (oz)).value() == Int (ni).value());
  assert (car (oz) == ni);
  assert (Double (cdr (oz)).value() == Double (nd).value());
  assert (cdr (oz) == nd);

  try {
    Number nnn (oz);
    println ("Expected exception didn't happen\n"); abort();
  } catch (Bad_Dynamic_Cast exc) {
    assert (std::string (exc.what())
	    == "Bad Dynamic Cast from Cons to Number");
  } catch (...) {
    println ("Unexpected exception\n"); abort();
  }

  try {
    car (Object (nd));
    println ("Expected exception didn't happen\n"); abort();
  } catch (Bad_Dynamic_Cast exc) {
    assert (std::string (exc.what())
	    == "Bad Dynamic Cast from Double to Cons");
  } catch (...) {
    println ("Unexpected exception...\n"); abort();
  }

  // Create garbage....
  //   for (int j=0; j < 10000000; ++j)
  //     Cons ni (Object (3), Double (3.2));
  //   sleep (30);
}

static void lispy_tests ()
{
  assert ("(1 2)"   == toString (cons (Object (1), cons (Object (2), nil))));
  assert ("(1 2)"   == toString (cons (1, cons (2, nil))));
  assert ("(1 . 2)" == toString (cons (1, 2)));

  Object x = list (1, 3.4, "foo", true);
  assert (toString (x) == "(1 3.4 foo 1)");
  assert (car (cdr (x)) == 3.4);
}

static void String_tests ()
{
  String s ("foo");
  assert (toString (s) == string ("foo"));
  assert (toString (Object ("foo")) == string ("foo"));
  assert (stringp (Object ("bar")));
  assert (stringp (s));
  assert (stringp (Object (s)));
}

static void Equals_tests ()
{
  int native_i = -14;
  Int i (native_i);
  double native_d = 2.718;
  Double d (native_d);
  Number ni (i);

  assert ((i.value() == i) && (i == i.value()) && (i.value() == native_i));
  assert ((d.value() == d) && (d == d.value()) && (d.value() == native_d));

  Int ix (2); ix = i; assert (ix == i);
  Number nx (i); nx = ni; assert ((nx == ni) && (nx == Int (ni)));

  assert (3 == Int (3));
  assert (3 == Number (3));
  assert (3 == Object (3));

  assert (Int (3) == 3);
  assert (Number (3) == 3);
  assert (Object (3) == 3);

  assert (Int (3) == Double (3.0));
  assert (Int (3) == Number (3));
  assert (Object  (3) == Number (3));
  assert (3 == Object (3L));
  assert (3 == Number (3L));
  assert (3.14 == Object (3.14));
  assert (3.14 == Number (3.14));
  assert (float (3.14) == Object (float (3.14)));
  assert (float (3.14) == Number (float (3.14)));
}

static void NumericPromotion_tests ()
{
  //assert ((is_same<NumericPromotion<int,int>::type, int>::value));
  assert ((is_same<NumericPromotion<int,Int>::type, Int>::value));
  assert ((is_same<NumericPromotion<Int,int>::type, Int>::value));
  assert ((is_same<NumericPromotion<Int,Int>::type, Int>::value));

  //assert ((is_same<NumericPromotion<int,double>::type, double>::value));
  assert ((is_same<NumericPromotion<int,Double>::type, Double>::value));
  assert ((is_same<NumericPromotion<Int,double>::type, Double>::value));
  assert ((is_same<NumericPromotion<Int,Double>::type, Double>::value));

  //assert ((is_same<NumericPromotion<double,int>::type, double>::value));
  assert ((is_same<NumericPromotion<Double,int>::type, Double>::value));
  assert ((is_same<NumericPromotion<double,Int>::type, Double>::value));
  assert ((is_same<NumericPromotion<Double,Int>::type, Double>::value));

  assert ((is_same<NumericPromotion<Int,Int>::type, Int>::value));
  assert ((is_same<NumericPromotion<Int,Number>::type, Number>::value));
  assert ((is_same<NumericPromotion<Number,Int>::type, Number>::value));
  assert ((is_same<NumericPromotion<Number,Number>::type, Number>::value));

  assert ((is_same<NumericPromotion<Int,Int>::type, Int>::value));
  assert ((is_same<NumericPromotion<Int,Object>::type, Number>::value));
  assert ((is_same<NumericPromotion<Object,Int>::type, Number>::value));
  assert ((is_same<NumericPromotion<Object,Object>::type, Number>::value));
}

// The key new idea is how to implement static overloading for smart references.

// The obvious attempt doesn't work:
static string ol1 (Object x) { return "Object"; }
static string ol1 (Number x) { return "Number"; }

static void static_overloading_1 ()
{
  Int x (42);
  assert (ol1 (Object (x)) == "Object"); // OK
  assert (ol1 (Number (x)) == "Number"); // OK
//assert (ol1 (Int    (x)) == "Number"); // COMPILE ERROR - ambiguous
}


// But this variation does
template <class T> static string ol2 (const IsA<T,Object>& x) { return "Object"; }
template <class T> static string ol2 (const IsA<T,Number>& x) { return "Number"; }

static void static_overloading_2 ()
{
  Int x (42);
  assert (ol2 (Object (x)) == "Object"); // OK
  assert (ol2 (Number (x)) == "Number"); // OK
  assert (ol2 (Int    (x)) == "Number"); // OK!!
}


// Now let's try standard idiomatic forwarding functions
static string ol3_Object (Object x) { return "Object"; }
static string ol3_Number (Number x) { return "Number"; }
template <class T> inline string ol3 (const IsA<T,Object>& x) { return ol3_Object (x); }
template <class T> inline string ol3 (const IsA<T,Number>& x) { return ol3_Number (x); }

static void static_overloading_3 ()
{
  Int x (42);
  assert (ol3 (Object (x)) == "Object"); // OK
  assert (ol3 (Number (x)) == "Number"); // OK
  assert (ol3 (Int    (x)) == "Number"); // OK!!
}


template<class T> inline std::string fun1 (const IsA<T,Object>& x)
{ return toString (~x*1); }
template<class T> inline std::string fun1 (const IsA<T,Number>& x)
{ return toString (~x*2); }
//template<class T> inline void fun1 (const IsA<T,Int>& x)  { println (*x*3); }

//template<class T> inline void fun1 (const IsA<T,Int>& x)  { println (*x*3); }

static void random_tests ()
{
  assert (fun1 (Object (3)) == "3");
  assert (fun1 (Number (3)) == "6");
  assert (fun1 (Int    (3)) == "6");

  Int i (3);
  Int i2 (7);
  Double d (4.1);
  Double d2 (2.718);
  Number ni (7);
  Number nd (10.1);
  Object oi (i);
  Object oci (3);
  Object od (d);
  Object ocd (3.14);
  Object x (null);
  assert (toString (x) == string ("null"));

  assert (numberp (i)  && intp (i));
  assert (numberp (d)  && doublep (d));
  assert (numberp (ni) &&  intp (ni) && !doublep (ni));
  assert (numberp (nd) && !intp (nd) && doublep (nd));

  ni = Number (4.4); assert (not intp (ni)); assert (Double (ni).value() == 4.4);
  assert ((i = Int (Number (5))).value() == 5);

  assert (Ob_convertible_p<Number> (Number (3)));
  assert (Ob_convertible_p<Number> (Int (3)));
  assert (not Ob_convertible_p<Int> (Number (3)));
  assert (not Ob_convertible_p<Int> (Double (3.3)));

  assert (Number (3).instanceof<Number>());
  assert (Int (3).instanceof<Number>());
  //assert (Int (3).instanceof<Double>()); // COMPILE ERROR
  assert (Int (3).instanceof<Object>());
  assert (! Object (3).instanceof<Double>());
  assert (Object (3).instanceof<Int>());
  assert (Object (3).instanceof<Number>());
}

// ----------------------------------------------------------------
// A minimal class
// TrivialString needs to define an operator<< to be boxed
struct TrivialString
{
  std::string s_;
  TrivialString (std::string s) : s_ (s) {}
};
inline std::ostream&
operator << (std::ostream& o, TrivialString myString)
{
  return o << myString.s_ << "!";
}

// ----------------------------------------------------------------
// A minimal class that wants to be a ISequence
// TrivialSequence needs to do a little more work to be integrated.
struct TrivialSequence
{
  int length () { return 3; }
};
inline std::ostream&
operator << (std::ostream& o, TrivialSequence)
{
  return o << "(TrivialSequence: 3)";
}

// TrivialSequence registers its "boxing traits"
namespace MObS
{
  // Give TrivialSequence a class name (optional)
  template <>
  struct ClassName<TrivialSequence>
  {
    static std::string Name () { return "TrivialSequence"; }
  };

  // Declare that a Boxed<TrivialSequence> is an ISequence, not just an Object
  template <> struct BoxingTraits<TrivialSequence>
    : public DefaultBoxingTraits<TrivialSequence>
  {
    typedef ISequence ObBase; // default is `Object'
  };

  // Need to define the virtual function `length' required by ISequence
  namespace ObI  // "Object Implementation"
  {
    template <>
    class Impl<Boxed<TrivialSequence> >
      : public DefaultImpl<Boxed<TrivialSequence> >
    {
    public:
      // Constructors are never inherited
      inline Impl (TrivialSequence v)
	: DefaultImpl<Boxed<TrivialSequence> > (v) {}
      // Typically implement virtual functions by forwarding
      virtual size_t length () { return value().length(); }
    };
  }
}

static void boxing_tests (void)
{
  {
    TrivialString myString ("MYSTRING");
    Object x1 = myString;
    Object x2 = Object (myString);
    Object x3 (myString);
    assert (typep<Boxed<TrivialString> >(x1));
    assert (toString (x3) == "MYSTRING!");
    assert (toString (Boxed<TrivialString>(x1).value()) == "MYSTRING!");
    assert (toString (Unbox<TrivialString>(x2)) == "MYSTRING!");

    Object pair = cons (3, TrivialString ("?"));
    assert (toString (pair) == "(3 . ?!)");
    assert (Unbox<TrivialString>(cdr (pair)).s_ == "?");
  }
  {
    TrivialSequence s;
    Object sob (s);
    Object j (1); j = sob;
    assert (j.getClass().getName() == "Boxed<TrivialSequence>");
    assert (length (sob) == 3);
    assert (toString (sob) == "(TrivialSequence: 3)");
    ISequence seq (s);
    assert (toString (seq) == "(TrivialSequence: 3)");
    String str ("foo");
    seq = str;
    seq = s;
    seq = str;
    seq = ISequence (s);
    // Int k (s);        // COMPILE ERROR
    // Int m (1); m = s; // COMPILE ERROR

    try {
      Int i (sob);
      println ("Expected exception didn't happen\n"); abort();
    } catch (Bad_Dynamic_Cast exc) {
      assert (std::string (exc.what())
	      == "Bad Dynamic Cast from Boxed<TrivialSequence> to Int");
    } catch (...) {
      println ("Unexpected exception...\n"); abort();
    }
  }
}

int main (int argc, char *argv)
{
  try {
    random_tests ();
    BinaryOps_tests ();
    Array_tests ();
    is_ObType_tests ();
    is_className_tests ();
    static_typep_tests ();
    Cons_tests();
    lispy_tests();
    String_tests();
    Equals_tests();
    NumericPromotion_tests ();
    boxing_tests ();
    static_overloading_1 ();
    static_overloading_2 ();
    static_overloading_3 ();
  } catch (const std::exception& ex) {
    std::cerr << "Exception occurred: " <<  ex.what() << "\n";
    return 1;
  } catch (...) {
    std::cerr << "Unexpected exception\n";
    return 2;
  }
  return 0;
}
