/* Copyright (C) Martin Buchholz 2003 */

/* Demo of MObS capabilities */

#include <cstdio>
#include <cassert>
#include <iostream>
#include <string>
#include "Object.hpp"
#include "NullType.hpp"
#include "Cons.hpp"
#include "lisp_y.hpp"
#include "Number.hpp"
#include "Int.hpp"
#include "Double.hpp"
#include "IO.hpp"
#include "toString.hpp"
#include "Number-arithmetic.hpp"
#include "IArray.hpp"
#include "Array.hpp"
#include "String.hpp"
#include "Boxed.hpp"
#include "BoxedBool.hpp"

using std::cout;
using std::string;
using namespace MObS;

// Type Hierarchy:
// IsABase
//   Object
//     NullType (null is only member)
//     Cons
//     Number (Abstract)
//       Int
//       Double
//     ISequence (Abstract)
//       String
//       IArray
//         Array<T>
//
//     Boxed<T>  (place in hierarchy depends on T)

// The object system can be used in a lispy way with all type-checking
// at run-time if all variables are declared as Object, or with full
// static type-checking if variables are never declared as Object.
// This is like Java (can always declare objects as Object in Java).

// Collections can be defined using static polymorphism like the STL
// or dynamic polymorphism like Java.

// But this hierarchy is extensible.
// Arbitrary foreign classes can be easily integrated into the framework by boxing.
// The class Object does not know about any of the other framework classes,

// E.g you have a pre-defined class Foo { ... },
//   there is an easy way of integrating this into the object system using boxing.
//     Then you can do ...

// Lisp-like code
static void lisp_y ()
{
  assert ("(1 foo)"   == toString (cons (1, cons ("foo", nil))));
  assert ("(1 . foo)" == toString (cons (1, "foo")));

  Object x = list (1, 3.4, "foo", true);
  assert ("(1 3.4 foo 1)" == toString (x));
  assert (car (cdr (x)) == 3.4);
}

// Java-like code
static void java_y ()
{
  Array<ISequence> seqs (4, String ("?"));
  seqs [0] = String ("foo"); // OK, a String is a Sequence
  seqs [1] = Array<int> (5); // OK, an Array is a Sequence
  // seqs [2] = Int (3);     // COMPILE ERROR, an Int is not a Sequence

  // Run-time polymorphism
  assert (seqs.length () == 4);
  assert (seqs[0].length () == 3);
  assert (seqs[0].length () == 5);
}

void BinaryOps_tests ();
void BinaryOps_tests ()
{
  assert (Object (2) * Int (3) == 6);
  assert (Object (2) * Double (.25) == .5);
  assert (Object (2) * Number (3) == 6);
  assert (Object (2) * Number (.25) == .5);
  assert (Object (2) * Object (3) == 6);
  assert (Object (2) * Object (.25) == .5);
  // Object (1) * Object ("foo"); // RUNTIME ERROR
}

// Arrays are generic.  Like Java builtin arrays.
// If the element type is specified at compile-time, no boxing happens.
// If you want Java-like collections of Objects, use Array<Ob>.

void Array_tests ();
void Array_tests ()
{
  Array<int> aa (3);
  assert (arrayp (aa));
  assert (sequencep (aa));

  assert (sequencep (Object (aa)));
  assert (sequencep (ISequence (aa)));
  assert (typep<Array<int> > (aa));
  aa.aset (1, 9);
  assert ((aa[0] = 11) == 11);
  assert (aa.aref (0) == aa[0]);
  assert (aa.aref (0) == 11);
  assert (toString (aa) == "[11, 9, 0]");
  assert (toString (Object (aa)) == toString (aa));

  Array<Object> a2 (7, null);
  assert (arrayp (a2));
  assert (typep<Array<Object> > (a2));
  a2[0] = String ("foo");
  a2[1] = true;  assert (  Unbox<bool> (a2[1]));
  a2[2] = false; assert (! Unbox<bool> (a2[2]));
  a2[3] = aa;
  a2[4] = Double (3.5);
  println (a2);
  assert (length (a2[0]) == 3);
}

void is_className_tests ();
void is_className_tests ()
{
#define CheckDynamicClassName(ob, s) \
assert (Object (ob).getClass().getName() == s)

  CheckDynamicClassName (Number (1), "Int");
  CheckDynamicClassName (Number (5.5), "Double");
  CheckDynamicClassName (Object (false), "Boxed<bool>");
  CheckDynamicClassName (Cons (nil, nil), "Cons");
  CheckDynamicClassName (null, "NullType");
  CheckDynamicClassName (Array<int>(3), "Array<i>"); // Compiler-dependent
  CheckDynamicClassName (String ("foo"), "String");
}

void Cons_tests ();
void Cons_tests ()
{
  Number ni (3);
  Number nd (3.34);
  Cons z (ni, nd);
  Object oz (z);
  assert (consp (z));
  assert (Consp (z));
  // consp (ni); // COMPILE ERROR
  // cdr (ni);   // COMPILE ERROR

  assert (consp (oz) && Consp (oz) && !numberp (oz));
  assert (! consp (Object (ni)) && ! Consp (Object (nd)));

  assert (Int (car (z)).value() == Int (ni).value());
  assert (car (z) == ni);
  assert (Car (z) == ni);
  assert (not (car (z) == nd));
  assert (Double (cdr (z)).value() == Double (nd).value());
  assert (cdr (z) == nd);
  assert (Cdr (z) == nd);
  assert (Int (car (oz)).value() == Int (ni).value());
  assert (car (oz) == ni);
  assert (Double (cdr (oz)).value() == Double (nd).value());
  assert (cdr (oz) == nd);

  try {
    Number nnn (oz);
    println ("Expected exception didn't happen\n"); abort();
  } catch (Bad_Dynamic_Cast exc) {
    assert (std::string (exc.what())
	    == "Bad Dynamic Cast from Cons to Number");
  } catch (...) {
    println ("Unexpected exception\n"); abort();
  }

  try {
    car (Object (nd));
    println ("Expected exception didn't happen\n"); abort();
  } catch (Bad_Dynamic_Cast exc) {
    assert (std::string (exc.what())
	    == "Bad Dynamic Cast from Double to Cons");
  } catch (...) {
    println ("Unexpected exception...\n"); abort();
  }

  // Create garbage....
  //   for (int j=0; j < 10000000; ++j)
  //     Cons ni (Object (3), Double (3.2));
  //   sleep (30);
}

template<class T> inline std::string fun1 (const IsA<T,Object>& x)
{ return toString (~x*1); }
template<class T> inline std::string fun1 (const IsA<T,Number>& x)
{ return toString (~x*2); }
//template<class T> inline void fun1 (const IsA<T,Int>& x)  { println (*x*3); }

void random_tests ();
void random_tests ()
{
  assert (fun1 (Object (3)) == "3");
  assert (fun1 (Number (3)) == "6");
  assert (fun1 (Int    (3)) == "6");

  Int i (3);
  Int i2 (7);
  Double d (4.1);
  Double d2 (2.718);
  Number ni (7);
  Number nd (10.1);
  Object oi (i);
  Object oci (3);
  Object od (d);
  Object ocd (3.14);
  Object x (null);
  assert (toString (x) == string ("null"));

  assert (numberp (i)  && intp (i));
  assert (numberp (d)  && doublep (d));
  assert (numberp (ni) &&  intp (ni) && !doublep (ni));
  assert (numberp (nd) && !intp (nd) && doublep (nd));

  ni = Number (4.4); assert (not intp (ni)); assert (Double (ni).value() == 4.4);
  assert ((i = Int (Number (5))).value() == 5);

  assert (Number (3).instanceof<Number>());
  assert (Int (3).instanceof<Number>());
  //assert (Int (3).instanceof<Double>()); // COMPILE ERROR
  assert (Int (3).instanceof<Object>());
  assert (! Object (3).instanceof<Double>());
  assert (Object (3).instanceof<Int>());
  assert (Object (3).instanceof<Number>());
}

// ----------------------------------------------------------------
// A minimal class
// TrivialString needs to define an operator<< to be boxed
struct TrivialString
{
  std::string s_;
  TrivialString (std::string s) : s_ (s) {}
};
inline std::ostream&
operator << (std::ostream& o, TrivialString myString)
{
  return o << myString.s_ << "!";
}

// ----------------------------------------------------------------
// A minimal class that wants to be a ISequence
// TrivialSequence needs to do a little more work to be integrated.
struct TrivialSequence
{
  int length () { return 3; }
};
inline std::ostream&
operator << (std::ostream& o, TrivialSequence)
{
  return o << "(TrivialSequence: 3)";
}

// TrivialSequence registers its "boxing traits"
namespace MObS
{
  // Give TrivialSequence a class name (optional)
  template <>
  struct ClassName<TrivialSequence>
  {
    static std::string Name () { return "TrivialSequence"; }
  };

  // Declare that a Boxed<TrivialSequence> is an ISequence, not just an Object
  template <> struct BoxingTraits<TrivialSequence>
    : public DefaultBoxingTraits<TrivialSequence>
  {
    typedef ISequence ObBase; // default is `Object'
  };

  // Need to define the virtual function `length' required by ISequence
  namespace ObI  // "Object Implementation"
  {
    template <>
    class Impl<Boxed<TrivialSequence> >
      : public DefaultImpl<Boxed<TrivialSequence> >
    {
    public:
      // Constructors are never inherited
      inline Impl (TrivialSequence v)
	: DefaultImpl<Boxed<TrivialSequence> > (v) {}
      // Typically implement virtual functions by forwarding
      virtual size_t length () { return value().length(); }
    };
  }
}

static void boxing_tests (void)
{
  {
    TrivialString myString ("MYSTRING");
    Object x1 = myString;
    Object x2 = Object (myString);
    Object x3 (myString);
    assert (typep<Boxed<TrivialString> >(x1));
    assert (toString (x3) == "MYSTRING!");
    assert (toString (Boxed<TrivialString>(x1).value()) == "MYSTRING!");
    assert (toString (Unbox<TrivialString>(x2)) == "MYSTRING!");
  }
  {
    TrivialSequence s;
    Object sob (s);
    Object j (1); j = sob;
    assert (j.getClass().getName() == "Boxed<TrivialSequence>");
    assert (length (sob) == 3);
    assert (toString (sob) == "(TrivialSequence: 3)");
    ISequence seq (s);
    assert (toString (seq) == "(TrivialSequence: 3)");
    String str ("foo");
    seq = str;
    seq = s;
    seq = str;
    seq = ISequence (s);
    // Int k (s);        // COMPILE ERROR
    // Int m (1); m = s; // COMPILE ERROR

    try {
      Int i (sob);
      println ("Expected exception didn't happen\n"); abort();
    } catch (Bad_Dynamic_Cast exc) {
      assert (std::string (exc.what())
	      == "Bad Dynamic Cast from Boxed<TrivialSequence> to Int");
    } catch (...) {
      println ("Unexpected exception...\n"); abort();
    }
  }
}

int main (int argc, char *argv)
{
  try {
    random_tests ();
    BinaryOps_tests ();
    Array_tests ();
    is_className_tests ();
    Cons_tests();
    lisp_y();
    java_y();
    boxing_tests ();
  } catch (const std::exception& ex) {
    std::cerr << "Exception occurred: " <<  ex.what() << "\n";
    return 1;
  } catch (...) {
    std::cerr << "Unexpected exception\n";
    return 2;
  }
  return 0;
}
