/* Copyright (C) Martin Buchholz 2003 */

#ifndef MObS_ObCast_hpp_INCLUDED_
#define MObS_ObCast_hpp_INCLUDED_

#include "config.hpp"
#include <stdexcept>
#include <string>
#include <boost/type_traits/conversion_traits.hpp>

namespace MObS
{
  class IsABase;
  template <class T, class Constraint> class IsA;
  template <class T> class StaticClass;

  // ----------------------------------------------------------------
  template <class From, class To>
  struct Ob_is_convertible
  {
    static const bool value =
    ::boost::is_convertible<IsA<From,From>&,IsA<From,To>& >::value;
  };

  template <class To, class From>
  inline bool Ob_convertible_p (const IsA<From,IsABase>& ob)
  {
    return Ob_is_convertible<From,To>::value;
  }

  // ----------------------------------------------------------------
  // bool typep<OBTYPE> (OBJECT)
  // Contrary to appearances, the typep conditional has THREE cases:
  // - static upcast    - compile-time constant TRUE.
  // - static downcast  - runtime dynamic check.
  // - static crosscast - commpile-time ERROR.
  // Thus, the code generated is always optimal, given a decent compiler.
  template <class To,
	    class From,
	    bool = Ob_is_convertible<From,To>::value,
	    bool = Ob_is_convertible<To,From>::value>
  struct static_typep_Selector;

  template <class To,
	    class From,
	    bool reverse_convertible>
  struct static_typep_Selector<To,From,true,reverse_convertible>
  {
    static const bool value = true;
  };

  template <class To, class From>
  struct static_typep_Selector<To,From,false,true>
  {
    static const bool value = false;
  };

  //Deliberately not implemented - we want compile error
  // template <class To, class From>
  // struct static_typep_Selector<To,From,false,false>

  template <class To, class From>
  inline bool static_typep (const IsA<From,IsABase>& ob)
  {
    return static_typep_Selector<To, From>::value;
  }

  // ----------------------------------------------------------------

  template <class To, class From,
	    bool static_typep = static_typep_Selector<To, From>::value>
  struct typep_Selector;

  template <class To, class From>
  struct typep_Selector<To, From, true>
  {
    static inline bool typep (const From& ob)
    {
      // Trivial conversion - no cast needed
      //printf ("Trivial cast\n");
      return 1;
    }
  };

  template <class To, class From>
  struct typep_Selector<To, From, false>
  {
    static inline bool typep (const From& ob)
    {
      // Dynamic cast required
      //printf ("Dynamic cast\n");
      return ob.template dynamic_instanceof<To>();
    }
  };

  // ----------------------------------------------------------------
  // Emulation of Lisp's `typep'
  template <class To, class From>
  inline bool typep (const IsA<From,IsABase>& ob)
  {
    return typep_Selector<To,From>::typep (static_cast<const From &>(ob));
  }

  // ----------------------------------------------------------------

  template <typename T>
  struct is_ObType
  {
    static const bool value = ::boost::is_convertible<T*,IsA<T,IsABase>*>::value;
  };

  class Bad_Dynamic_Cast : public std::exception
  {
  private:
    std::string description;
  public:
    Bad_Dynamic_Cast () : description ("Bad Dynamic Cast") {}

    Bad_Dynamic_Cast (std::string from, std::string to)
      : description (std::string ("Bad Dynamic Cast from ")
		     + from + " to " + to)
    {}

    virtual const char * what() const throw() { return description.c_str(); }
    ~Bad_Dynamic_Cast () throw () {}
  };

  template <class To>
  class Bad_Dynamic_Cast_To : public Bad_Dynamic_Cast
  {
  public:
    template <class From>
    Bad_Dynamic_Cast_To (const IsA<From,IsABase>& o)
      : Bad_Dynamic_Cast (static_cast<const From&>(o).getClass().getName(),
			  StaticClass<To>::getName())
    {}
  };

//   // Unsafe!!  Consider using Ob_dynamic_cast instead.
//   template <class To, class From>
//   inline To
//   Ob_reinterpret_cast (const From& ob)
//   {
//     return To (reinterpret_cast<const To&> (ob));
//   }

  // Semantics of Ob_dynamic_cast are as follows:
  // FromObClass and ToObClass must be
  // If the cast would be trivial (upcast), the cast is performed statically (free).

  // If the cast is a downcast, then a regular dynamic_cast is performed
  // on the underlying pointers, and an Error thrown at run-time.

  // If the classes are non-related, then an (inscrutable) compile-time error happens.
  // Example:
  // Cons c (...);
  // Int i (...);
  // Object oc (c);
  // Object oi (i);
  // Object_dynamic_cast<Cons>(i)  ==> compile-time error
  // Object_dynamic_cast<Cons>(oi) ==> run-time error
  // Object_dynamic_cast<Cons>(oc) ==> run-time successful cast
  // Object_dynamic_cast<Cons>(c)  ==> trivially successful (free) cast


//   template <class To, class From>
//   inline To
//   Object_dynamic_cast (const From& ob)
//   {
//     if (typep<To> (ob))
//       return To (reinterpret_cast<const To&> (ob));
//     else
//       throw Bad_Dynamic_Cast_To<To> (ob);
//   }

  template <class ObClass>
  class Maybe
  {
    const void* p;

  public:
    template <class ObClass2>
    Maybe (const IsA<ObClass2,IsABase>& ob)
      : p ((typep<ObClass>(ob)) ? &ob : 0) {}

    operator ObClass () const
    {
      assert (p != 0);
      return ObClass (*(reinterpret_cast<const ObClass *>(p)));
    }

    const ObClass * operator-> () const
    {
      assert (p != 0);
      return reinterpret_cast<const ObClass *>(p);
    }

    // ----------------------------------------------------------------
    // really operator bool()
    //operator const void * () const { return p; }

    // From "C++ Templates" p.392
  private:
    struct BoolConversionSupport { int dummy; };

  public:
    operator int BoolConversionSupport::* () const
    {
      return p ? &BoolConversionSupport::dummy : 0;
    }
  };

} // namespace MObS

#endif // Recursive Inclusion Guard
