/* Copyright (C) Martin Buchholz 2003 */

#ifndef MObS_Number_arithmetic_hpp_INCLUDED_
#define MObS_Number_arithmetic_hpp_INCLUDED_

#ifndef INT_DIVISION_PROMOTES
#define INT_DIVISION_PROMOTES 1
#endif

#include <boost/type_traits/arithmetic_traits.hpp>
#include <boost/type_traits/same_traits.hpp>
#include "Object.hpp"
#include "Number.hpp"
#include "Int.hpp"
#include "Double.hpp"

namespace MObS
{

  // ----------------------------------------------------------------
  template <typename T, typename U> struct NumericPromotion;

  namespace ObI
  {
    namespace NumberOps
    {

      // Compute a meaningful "Type Category" for further template hackery
      enum TypeCategory
	{
	  NativeNumericType,
	  ConcreteNumericObType,
	  AbstractNumericObType,
	  NonNumericObType
	};

      template <typename T,
		bool = ::boost::is_arithmetic<T>::value,
		bool = is_ObType<T>::value,
		bool = is_ConcreteNumericObType<T>::value,
		bool = ::boost::is_convertible<T*,IsA<T,Number>*>::value
      > struct typeCategory;
      template <typename T>
      struct typeCategory<T, true, false, false, false>
      {
	static const TypeCategory value = NativeNumericType;
      };
      template <typename T, bool any>
      struct typeCategory<T, false, true, true, any>
      {
	static const TypeCategory value = ConcreteNumericObType;
      };
      template <typename T>
      struct typeCategory<T, false, true, false, true>
      {
	static const TypeCategory value = AbstractNumericObType;
      };
      template <typename T>
      struct typeCategory<T, false, true, false, false>
      {
	static const TypeCategory value = NonNumericObType;
      };

      template <typename T, TypeCategory T_category,
		typename U, TypeCategory U_category>
      struct NumericPromotion_1;

    } // namespace NumberOps
  } // namespace ObI

  // ----------------------------------------------------------------

  template <typename T, typename U>
  struct NumericPromotion
  {
    typedef typename
    ObI::NumberOps::NumericPromotion_1
    <T, ObI::NumberOps::typeCategory<T>::value,
     U, ObI::NumberOps::typeCategory<U>::value>::type type;
  };

  template <>
  struct NumericPromotion<Int, Double>
  {
    typedef Double type;
  };

  template <>
  struct NumericPromotion<Double, Int>
  {
    typedef Double type;
  };

  namespace ObI
  {
    namespace NumberOps
    {

      // Number ^ Any -> Number always
      template <class T, typename U, TypeCategory AnyType>
      struct NumericPromotion_1<T, AbstractNumericObType,
				U, AnyType>
      {
	typedef Number type;
      };

      // dynamic_cast to Number ^ Any -> Number always
      template <class T, typename U, TypeCategory AnyType>
      struct NumericPromotion_1<T, NonNumericObType,
				U, AnyType>
      {
	typedef Number type;
      };

      // native ^ native
      // convert to ConcreteObType's, promote, convert back to native!
      template <class T, typename U>
      struct NumericPromotion_1<T, NativeNumericType,
				U, NativeNumericType>
      {
	typedef typename Native2Concrete<T>::type T_concrete;
	typedef typename Native2Concrete<U>::type U_concrete;
	typedef typename NumericPromotion<T_concrete, U_concrete>::type concrete;
	typedef typename Concrete2Native<concrete>::type type;
      };

      // native ^ concrete
      // convert native to concrete; recurse
      template <class T, typename U>
      struct NumericPromotion_1<T, NativeNumericType,
				U, ConcreteNumericObType>
      {
	typedef typename Native2Concrete<T>::type T_concrete;
	typedef typename NumericPromotion<T_concrete, U>::type type;
      };

      // native ^ NumberType -> Number always
      template <class T, typename U>
      struct NumericPromotion_1<T, NativeNumericType,
				U, AbstractNumericObType>
      {
	typedef Number type;
      };

      // native ^ NonNumericObType -> Number always
      template <class T, typename U>
      struct NumericPromotion_1<T, NativeNumericType,
				U, NonNumericObType>
      {
	typedef Number type;
      };

      // concrete ^ NonNumericObType -> Number always
      template <class T, typename U>
      struct NumericPromotion_1<T, ConcreteNumericObType,
				U, NonNumericObType>
      {
	typedef Number type;
      };

      // concrete ^ NumberType -> Number always
      template <class T, typename U>
      struct NumericPromotion_1<T, ConcreteNumericObType,
				U, AbstractNumericObType>
      {
	typedef Number type;
      };

      // concrete ^ Native -> reduce to concrete ^ concrete
      template <class T, typename U>
      struct NumericPromotion_1<T, ConcreteNumericObType,
				U, NativeNumericType>
      {
	typedef typename Native2Concrete<U>::type U_concrete;
	typedef typename NumericPromotion<T, U_concrete>::type type;
      };

      // concrete ^ concrete
      // In general, we have to explicitly specify these,
      // but we can do the "main diagonal" automatically.
      // This assumes you don't want Int / Int ==> Double
      // This whole class of conversions can be automated if we use typeof().
      template <class T>
      struct NumericPromotion_1<T, ConcreteNumericObType,
				T, ConcreteNumericObType>
      {
	typedef T type;
      };

      template <class Op> struct BinOp;

      // ----------------------------------------------------------------
      // Template algorithms want to deal with Types, not operators;
      // so convert each operator to an equivalent Functor.

      namespace ArithmeticBinOp {
	template <typename T, typename U>
	struct ReturnType
	{
	  typedef typename NumericPromotion<T,U>::type type;
	};
      } // namespace ArithmeticBinOp

      namespace ComparisonBinOp {
	template <typename T, typename U>
	struct ReturnType
	{
	  typedef bool type;
	};
      } // namespace ComparisonBinOp

      struct ComparisonReturnType
      {
	template <typename T, typename U>
	struct ReturnType
	{
	  typedef bool type;
	};
      };

      struct ArithmeticReturnType
      {
	template <typename T, typename U>
	struct ReturnType
	{
	  typedef typename ArithmeticBinOp::ReturnType<T,U>::type type;
	};
      };

      struct Adder : public ArithmeticReturnType
      {
	template <typename T, typename U>
	static inline typename ReturnType<T,U>::type
	Doit (T o1, U o2)
	{
	  return o1 + o2;
	}
      };

      struct Subtracter : public ArithmeticReturnType
      {
	template <typename T, typename U>
	static inline typename ReturnType<T,U>::type
	Doit (T o1, U o2)
	{
	  return o1 - o2 ;
	}
      };

      struct Multiplier : public ArithmeticReturnType
      {
	template <typename T, typename U>
	static inline typename ReturnType<T,U>::type
	Doit (T o1, U o2)
	{
	  return o1 * o2;
	}
      };

      struct Divider
      {

	template <typename T, typename U>
	struct ReturnType
	{
	  typedef typename NumericPromotion<T,U>::type type;
	};

#if INT_DIVISION_PROMOTES
	template <typename U>
	struct ReturnType<int, U>
	{
	  typedef typename ReturnType<double,U>::type type;
	};

	template <typename U>
	struct ReturnType<Int, U>
	{
	  typedef typename ReturnType<Double,U>::type type;
	};

	template <typename U>
	static inline typename ReturnType<int,U>::type
	Doit (int o1, U o2)
	{
	  return Doit (static_cast<double>(o1), o2);
	}

	template <class T, typename U>
	static inline typename ReturnType<Int,U>::type
	Doit (const IsA<T,Int>& o1, U o2)
	{
	  return Doit (static_cast<double>(o1->value()), o2);
	}
#endif /* INT_DIVISION_PROMOTES */

	template <typename T, typename U>
	static inline typename ReturnType<T,U>::type
	Doit (T o1, U o2)
	{
	  return o1 / o2;
	}

      }; // Divider


      struct Equals : public ComparisonReturnType
      {
	template <typename T, typename U>
	static inline typename ReturnType<T,U>::type // always bool, actually
	Doit (T o1, U o2)
	{
	  return o1 == o2;
	}
      };

      struct NotEquals : public ComparisonReturnType
      {
	template <typename T, typename U>
	static inline typename ReturnType<T,U>::type // always bool, actually
	Doit (T o1, U o2)
	{
	  return o1 != o2;
	}
      };

      // ----------------------------------------------------------------

      // The guts of the implementation of all binary operators.

      template <class Op> struct BinOp
      {
	template <typename T, TypeCategory T_category,
		  typename U, TypeCategory U_category>
	struct Selector;

	// Reduce first arg to native; then second arg to native; then use native Op

	// concrete ^ ANYTHING -> native ^ ANYTHING
	template <typename T,
		  typename U, TypeCategory AnyUType>
	struct Selector<T, ConcreteNumericObType,
			U, AnyUType>
	{
	  typedef typename Op::ReturnType<T,U>::type ReturnType;
	  static inline ReturnType
	  Doit (const T& o1, const U& o2)
	  {
	    return ReturnType (Op::Doit (o1.value(), o2));
	  }
	};

	// Number ^ ANYTHING -> native ^ ANYTHING
	template <typename T,
		  typename U, TypeCategory AnyUType>
	struct Selector<T, AbstractNumericObType,
			U, AnyUType>
	{
	  typedef typename Op::ReturnType<T,U>::type ReturnType;
	  static ReturnType
	  Doit (const T& o1, const U& o2)
	  {
	    if (Maybe<Int> i = o1)
	      return ReturnType (Op::Doit (i->value(), o2));
	    else if (Maybe<Double> d = o1)
	      return ReturnType (Op::Doit (d->value(), o2));
	    else
	      abort();
	  }
	};

	// polymorphic ^ ANYTHING -> Number ^ ANYTHING
	template <typename T,
		  typename U, TypeCategory AnyUType>
	struct Selector<T, NonNumericObType,
			U, AnyUType>
	{
	  typedef typename Op::ReturnType<T,U>::type ReturnType;
	  static ReturnType
	  Doit (const T& o1, const U& o2)
	  {
	    return ReturnType (Op::Doit (Number (o1), o2));
	  }
	};

	// native ^ concrete
	template <typename T, typename U>
	struct Selector<T, NativeNumericType,
			U, ConcreteNumericObType>
	{
	  typedef typename Op::ReturnType<T,U>::type ReturnType;
	  static inline ReturnType
	  Doit (const T& o1, const U& o2)
	  {
	    return ReturnType (Op::Doit (o1, o2.value()));
	  }
	};

	// native ^ Number
	template <typename T, typename U>
	struct Selector<T, NativeNumericType,
			U, AbstractNumericObType>
	{
	  typedef typename Op::ReturnType<T,U>::type ReturnType;
	  static ReturnType
	  Doit (const T& o1, const U& o2)
	  {
	    if (Maybe<Int> i = o2)
	      return ReturnType (Op::Doit (o1, i->value()));
	    else if (Maybe<Double> d = o2)
	      return ReturnType (Op::Doit (o1, d->value()));
	    else
	      abort();
	  }
	};

	// native ^ dynamic_cast to Number
	template <typename T, typename U>
	struct Selector<T, NativeNumericType,
			U, NonNumericObType>
	{
	  typedef typename Op::ReturnType<T,U>::type ReturnType;
	  static ReturnType
	  Doit (const T& o1, const U& o2)
	  {
	    return ReturnType (Op::Doit (o1, Number (o2)));
	  }
	};

	// native ^ native
	template <typename T, typename U>
	struct Selector<T, NativeNumericType,
			U, NativeNumericType>
	{
	  typedef typename Op::ReturnType<T,U>::type ReturnType;
	  static inline ReturnType
	  Doit (const T& o1, const U& o2)
	  {
	    return ReturnType (Op::Doit (o1, o2));
	  }
	};

	template <typename T, typename U>
	static inline typename Op::ReturnType<T,U>::type
	Doit (const T& o1, const U& o2)
	{
	  return Selector <
	    T, typeCategory<T>::value,
	    U, typeCategory<U>::value
	    >::Doit (o1, o2);
	}
      };

      // We must use compile-time filtering to avoid template instantiation
      // failures for types we don't know about.
      // This relies on a gray part of the standard - that return types
      // participate in "template argument deduction".
      template <class T, class U, class Op,
		bool =
		(::boost::is_same<T,Object>::value ||
		 ::boost::is_same<U,Object>::value ||
		 ::boost::is_convertible<T*,IsA<T,Number>*>::value ||
		 ::boost::is_convertible<U*,IsA<U,Number>*>::value)>
// 		(is_ObType<T>::value ||
// 			is_ObType<U>::value)>
      struct ReturnType;

      template <class T, class U, class Op>
      struct ReturnType<T,U,Op,true>
      {
	typedef typename Op::ReturnType<T,U>::type type;
      };

    } // namespace NumberOps
  } // namespace ObI

  // We avoid having to write reams of mind-numbing code by:
  // - using a trivial Functor for every binop,
  //   and doing all computations in binop-independent way.

  template <typename T, typename U>
  inline typename ObI::NumberOps::ReturnType<T,U,ObI::NumberOps::Adder>::type
  operator+ (T o1, U o2)
  {
    using namespace ObI::NumberOps;
    return BinOp<Adder>::Doit (o1, o2);
  }

  template <class T, typename U>
  inline typename ObI::NumberOps::ReturnType<T,U,ObI::NumberOps::Subtracter>::type
  operator- (T o1, U o2)
  {
    using namespace ObI::NumberOps;
    return BinOp <Subtracter>::Doit (o1, o2);
  }

  template <class T, typename U>
  inline typename ObI::NumberOps::ReturnType<T,U,ObI::NumberOps::Multiplier>::type
  operator* (T o1, U o2)
  {
    using namespace ObI::NumberOps;
    return BinOp <Multiplier>::Doit (o1, o2);
  }

  template <class T, typename U>
  inline typename ObI::NumberOps::ReturnType<T,U,ObI::NumberOps::Divider>::type
  operator/ (T o1, U o2)
  {
    using namespace ObI::NumberOps;
    return BinOp <Divider>::Doit (o1, o2);
  }

  // ----------------------------------------------------------------

  template <class T, typename U>
  inline typename ObI::NumberOps::ReturnType<T,U,ObI::NumberOps::Equals>::type
  operator== (T o1, U o2)
  {
    using namespace ObI::NumberOps;
    return BinOp <Equals>::Doit (o1, o2);
  }

  template <class T, typename U>
  inline typename ObI::NumberOps::ReturnType<T,U,ObI::NumberOps::NotEquals>::type
  operator!= (T o1, U o2)
  {
    return ! (o1 == o2);
  }

  // ----------------------------------------------------------------

} // namespace MObS

#endif // Recursive Inclusion Guard
