/* Copyright (C) Martin Buchholz 2003 */

import java.io.*;
import java.util.regex.*;
import martin.Command;
import martin.CommandResults;

public class CompileErrors
{
    private static final String standard_prelude_ =
	"#include <iostream>\n" +
	"#include <cassert>\n"  +
	"#include <string>\n"   +
	"#include \"Object.hpp\"\n" +
	"#include \"NullType.hpp\"\n" +
	"#include \"Cons.hpp\"\n" +
	"#include \"Number.hpp\"\n" +
	"#include \"Double.hpp\"\n" +
	"#include \"Int.hpp\"\n" +
	"#include \"toString.hpp\"\n" +
	"#include \"IO.hpp\"\n" +
	"#include \"IArray.hpp\"\n" +
	"#include \"String.hpp\"\n" +
	"#include \"Boxed.hpp\"\n" +
	"#include \"BoxedBool.hpp\"\n" +
	"using namespace MObS;\n" +
	"using std::cout;\n" +
	"using std::string;\n" +
	"\n";

    private static final String prog_    = "CompileErrors";
    private static final String prog_cc_ = prog_ + ".cc";

    private static void WriteStringToFile (String filename, String contents)
	throws FileNotFoundException, IOException {
	PrintWriter pw = new PrintWriter
	    (new FileOutputStream (new File (filename)));
	pw.print (contents);
	pw.close ();
    }

    private static void CheckGccErrorMessage (String fileContents,
					      String errorMsgRegex)
	throws FileNotFoundException, IOException
    {
	WriteStringToFile (prog_cc_, fileContents);
	String [] argv = new String []
	    { "g++", "-Wall",
	      "-fmessage-length=0", // Avoid unpredictable line breaks
	      "-I.", "-I../../../boost_1_29_0", "-I/usr/local/include",
	      "-L/usr/local/lib", "-lgc", "-ldl",
	      "-Wl,--rpath", "-Wl,/usr/local/lib",
	      prog_cc_,
	      "-L.", "-lMObS",
	      "-o", prog_ };

	CommandResults res = Command.run (argv);

	if (res.exception == null &&
	    res.exitVal != 0 &&
	    Pattern.compile (errorMsgRegex).matcher (res.stderr).find()) {
	    System.out.println ("PASS");
	} else {
	    System.out.println ("\nTEST FAILED:");
	    System.out.println ("Regex=\"" + errorMsgRegex + "\"");
	    res.debugPrint ();
	}

	new File (prog_cc_).delete();
	new File (prog_).delete();
    }

    private static void StandardAppTest (String testCode,
					 String errorMsgRegex)
	throws FileNotFoundException, IOException
    {
	StringBuffer s = new StringBuffer (standard_prelude_);
	s.append (testCode);
	CheckGccErrorMessage (s.toString(), errorMsgRegex);
    }

    private static void MainBodyTest (String mainBody,
				      String errorMsgRegex)
	throws FileNotFoundException, IOException
    {
	StandardAppTest ("int main (int argc, char *argv[])\n{\n" +
			 mainBody +
			 "\nreturn 0;\n}\n",
			 errorMsgRegex);
    }

    private static void runTests ()
	throws FileNotFoundException, IOException
    {
	MainBodyTest ("return x9;", ".x9. undeclared");
	MainBodyTest ("Int y (\"3\");", "invalid conversion");
	MainBodyTest ("Cons x (Int (3));", "invalid static_cast");
	MainBodyTest ("consp (Number (3));", "incomplete type .*static_typep_Selector");
	MainBodyTest ("cdr (Number (3));", "incomplete type .*static_typep_Selector");
	MainBodyTest ("ISequence x (Int (3));", "invalid static_cast");
	MainBodyTest ("ISequence x (String (\"foo\")); x = Object (3);", "no match for");

	{
	    StringBuffer s = new StringBuffer ();
	    s.append ("static void ol1 (Object x) {}\n");
	    s.append ("static void ol1 (Number x) {}\n");
	    s.append ("int main (int argc,char *argv[])\n{\n");
	    s.append ("ol1 (Int (43));\n}\n");
	    StandardAppTest (s.toString(), "call of overloaded.* is ambiguous");
	}
    }

    public static void main (String []args) {
	try {
	    runTests ();
	} catch (Exception e) {
	    e.printStackTrace(); Runtime.getRuntime().exit (1);
	}
    }
}
